// IActiveScriptSite.c
//
// This file contains our MyRealIActiveScriptSite object. This is
// a multiple interface object that has a standard IActiveScriptSite
// as the base object, and a standard IActiveScriptSiteWindow
// sub-object.
//
// Our IActiveScriptSite's GetItemInfo function knows about, and
// returns, only one automation object that we allow the script to
// use via the name "application". This causes a pointer to our IApp
// (actually MyRealIApp) object to be returned.

#include <windows.h>
#include <stddef.h>
#include <activscp.h>
#include "IActiveScriptSite.h"
#include "AppObject.h"
#include "Guids.h"
#include "Extern.h" 

// Since we're going to support using only 1 script engine at a time,
// we need only one IActiveScriptSite. We'll declare it globally to
// make this easy (instead of GlobalAlloc'ing it)
class MyRealIActiveScriptSite	MyActiveScriptSite;

// For OnHandleError
static const WCHAR				LineNumStr[] = {'L','i','n','e',' '};

// For uNumToAscii()
#define NUMDIGITS 9
static const unsigned long ConvChars[NUMDIGITS] = {1000000000, 100000000, 10000000, 1000000, 100000, 10000, 1000, 100, 10};













//===========================================================================
//======================= IActiveScriptSite functions =======================
//===========================================================================

STDMETHODIMP MyRealIActiveScriptSite::QueryInterface(REFIID riid, void **ppv)
{
	// An ActiveX Script Host is supposed to provide an object
	// with multiple interfaces, where the base object is an
	// IActiveScriptSite, and there is an IActiveScriptSiteWindow
	// sub-object. Therefore, a caller can pass an IUnknown or
	// IActiveScript VTable GUID if he wants our IActiveScriptSite.
	// Or, the caller can pass a IActiveScriptSiteWindow VTable
	// GUID if he wants our IActiveScriptSiteWindow sub-object
	if (IsEqualIID(riid, IID_IUnknown) || IsEqualIID(riid, IID_IActiveScriptSite))
		*ppv = static_cast<IActiveScriptSite *>(this);
	else if (IsEqualIID(riid, IID_IActiveScriptSiteWindow))
		*ppv = static_cast<IActiveScriptSiteWindow *>(this); 
	else
	{
		*ppv = 0;
		return(E_NOINTERFACE);
	}
	
	this->AddRef();
	return(S_OK);
}

STDMETHODIMP_(ULONG) MyRealIActiveScriptSite::AddRef()
{
	// Since our MyRealIActiveScriptSite is not allocated, but instead
	// declared statically, we have no need to maintain a reference
	// count
	return(1);
}

STDMETHODIMP_(ULONG) MyRealIActiveScriptSite::Release()
{
	// Since our MyRealIActiveScriptSite is not allocated, but instead
	// declared statically, we have no need to maintain a reference
	// count, nor free it.
	//
	// NOTE: If our MyRealIActiveScriptSite did contain some resources
	// that we had to free, then we'd need to maintain a reference
	// count, and free those resources when the reference count was
	// zero
	return(1);
}

// Called by the script engine to get any pointers to our own host-defined objects whose
// functions a script may directly call. For example, maybe we've called the engine's
// IActiveScript->AddNamedItem() to define an object named "application". In a VBscript
// we run, there may be a line such as:
//
// result = application.DoSomething()
//
// In this case, the engine will call our GetItemInfo to get our ITypeInfo object for
// our own IApp object. The engine will use this to find out what type args our
// IApp object's DoSomething function is passed (if any), and what type of data it
// returns. Then, the engine will call our GetItemInfo again to get a pointer to our
// actual IApp object and use its (IDispatch) GetIDsOfNames function to get the
// DISPID for its DoSomething function, and call its Invoke function to indirectly
// call DoSomething.
STDMETHODIMP MyRealIActiveScriptSite::GetItemInfo(LPCOLESTR objectName, DWORD dwReturnMask, IUnknown **objPtr, ITypeInfo **typeInfo)
{
	register HRESULT	hr;

	// Assume failure
	hr = E_FAIL;
	if (dwReturnMask & SCRIPTINFO_IUNKNOWN) *objPtr = 0;
	if (dwReturnMask & SCRIPTINFO_ITYPEINFO) *typeInfo = 0;

   // Does the engine want our IApp object (has the string name "application")?
	if (!lstrcmpiW(objectName, &MyRealIApp.MyAppObjectName[0]))
	{
		// Does the engine want a pointer to our IApp object returned?
		if (dwReturnMask & SCRIPTINFO_IUNKNOWN)
		{
			// Give the engine a pointer to our IApp object. Engine will call
			// its AddRef() function, and later Release() it when done
			*objPtr = getAppObject();
		}

		// Does the engine want the ITypeInfo for our IApp object returned?
		if (dwReturnMask & SCRIPTINFO_ITYPEINFO)
		{
			// Make sure we have an ITypeInfo for our IApp object. (The script
			// engine needs to figure out what args are passed to our IApp's
			// extra functions, and what those functions return. And for that,
			// it needs IApp's ITypeInfo). The engine will call its AddRef()
			// function, and later Release() it when done
			if ((hr = getAppObjectITypeInfo(typeInfo))) goto bad;
		}

		hr = S_OK;
	}
bad:
	return(hr);
}

/*************************** uNumToAscii() ***********************
 * Converts the passed ULONG to an ascii string of digits in base 10,
 * putting that string into the passed buffer, left-justified.
 *
 * num =	unsigned long value to convert
 * buffer =	pointer to buffer into which to place the ascii chars. Must
 *			be at least 16 chars wide.
 *
 * RETURNS: Pointer to end of the string (ie, where null byte would be
 * stored if string is to be nul-terminated).
 */

static WCHAR * uNumToAscii(unsigned long num, WCHAR * buffer)
{
	register const unsigned long *	conv = &ConvChars[0];
	register unsigned long			div;
	unsigned char					flg;
	unsigned char					i;

	// Haven't got first non-0 digit yet
	flg = 0;
	i = NUMDIGITS;

	// Skip leading 0's
	while (num < *conv)
	{
		++conv;
		--i;
	}

	// Convert digits
	while (num > 9)
	{
		div = *(conv)++;
		*(buffer)++ = flg = (unsigned char)(num/div) + '0';
		num %= div;
		--i;
	}

	// See how many 0 places we need to add
	if (flg)
	{
		while (i--) *(buffer)++ = '0';
	}

	// Convert 1's digit and store
	*(buffer)++ = (unsigned char)num + '0';

	// Null-terminate
//	*buffer = 0;

	// Return end of string
	return(buffer);
}

// Called by the script engine when there is an error running/parsing a script.
// The script engine passes us its IActiveScriptError object whose functions
// we can call to get information about the error, such as the line/character
// position (in the script) where the error occurred, an error message we can
// display to the user, etc. Typically, our OnScriptError will get the error
// message and display it to the user.
STDMETHODIMP MyRealIActiveScriptSite::OnScriptError(IActiveScriptError *scriptError)
{
	ULONG				lineNumber;
	BSTR				desc;
	EXCEPINFO			ei;
	register WCHAR		*msg;
	register WCHAR		*ptr;
	register DWORD		len;

	// Call GetSourcePosition() to retrieve the line # where
	// the error occurred in the script
	scriptError->GetSourcePosition(0, &lineNumber, 0);

	// Call GetSourceLineText() to retrieve the line in the script that
	// has an error.
	//
	// Note: The IActiveScriptError is supposed to clear our BSTR pointer
	// if it can't return the line. So we shouldn't have to do
	// that first. But you may need to uncomment the following line
	// if a script engine isn't properly written. Unfortunately, too many
	// engines are not written properly, so we uncomment it
	desc = 0;
	scriptError->GetSourceLineText(&desc);

	// Call GetExceptionInfo() to fill in our EXCEPINFO struct with more
	// information.
	//
	// Note: The IActiveScriptError is supposed to zero out any fields of
	// our EXCEPINFO that it doesn't fill in. So we shouldn't have to do
	// that first. But you may need to uncomment the following line
	// if a script engine isn't properly written
//	ZeroMemory(&ei, sizeof(ei));
	scriptError->GetExceptionInfo(&ei);

	// Allocate a buffer to format the message
	len = (24*sizeof(WCHAR)) + sizeof(LineNumStr) + 2;	// leave room for the line number + "Line " + terminating nul
	if (ei.bstrSource) len += ::SysStringByteLen(ei.bstrSource) + 4;
	if (ei.bstrDescription) len += ::SysStringByteLen(ei.bstrDescription) + 4;
	if (desc) len += ::SysStringByteLen(desc) + 4;

	if ((msg = ptr = (WCHAR *)::GlobalAlloc(GMEM_FIXED, len)))
	{
		// Format the message we'll display to the user
		if (ei.bstrSource)
		{
			len = ::SysStringLen(ei.bstrSource);
			CopyMemory(ptr, ei.bstrSource, len * sizeof(WCHAR));
			ptr += len;
			*(ptr)++ = '\r';
			*(ptr)++ = '\n';
		}

		CopyMemory(ptr, LineNumStr, sizeof(LineNumStr));
		ptr += (sizeof(LineNumStr)/sizeof(WCHAR));
		ptr = uNumToAscii(lineNumber + 1, ptr);

		if (ei.bstrDescription)
		{
			*(ptr)++ = ':';
			*(ptr)++ = ' ';
			len = ::SysStringLen(ei.bstrDescription);
			CopyMemory(ptr, ei.bstrDescription, len * sizeof(WCHAR));
			ptr += len;
		}

		if (desc)
		{
			*(ptr)++ = '\r';
			*(ptr)++ = '\n';
			len = ::SysStringLen(desc);
			CopyMemory(ptr, desc, len * sizeof(WCHAR));
			ptr += len;
		}

		*ptr = 0;
		::PostMessage(MainWindow, WM_APP, (WPARAM)msg, 0);
	}

	// Free what we got from the IActiveScriptError functions
	::SysFreeString(desc);
	::SysFreeString(ei.bstrSource);
	::SysFreeString(ei.bstrDescription);
	::SysFreeString(ei.bstrHelpFile);

	return(S_OK);
}

// Called when the script engine wants to know what language ID our
// program is using.
STDMETHODIMP MyRealIActiveScriptSite::GetLCID(LCID *lcid)
{
	*lcid = LOCALE_USER_DEFAULT;
	return(S_OK);
}

// Called when the script engine wishes to retrieve the version number
// (as a string) for the current document. We are expected to return a
// SysAllocString()'ed BSTR copy of this version string. The engine
// typically uses this string to make sure the internally-cached state
// of the script engine is still in sync with the current document. The
// engine may get this string when we ParseScriptText some script, and
// then once again retrieve this string before running the script to
// ensure that we haven't updated the script in the interim. (Presumably
// we would update this version string if we did that).
STDMETHODIMP MyRealIActiveScriptSite::GetDocVersionString(BSTR *version) 
{
	*version = SysAllocString(L"");
	return(S_OK);
}

// Called when the engine has completely finished running scripts and
// is returning to INITIALIZED state. In many engines, this is not
// called because an engine alone can't determine when we are going
// to stop adding scripts to it.
STDMETHODIMP MyRealIActiveScriptSite::OnScriptTerminate(const VARIANT *pvr, const EXCEPINFO *pei)
{
	return(S_OK);
}

// Called when the script engine's state is changed (for example, by us calling
// the engine IActiveScript->SetScriptState). We're passed the new state.
STDMETHODIMP MyRealIActiveScriptSite::OnStateChange(SCRIPTSTATE state)
{
	return(S_OK);
}

// Called right before the script engine executes/interprets each script
// added via the engine IActiveScriptParse->ParseScriptText() or
// AddScriptlet(). This is also called when our IApp object calls
// some function in the script.
STDMETHODIMP MyRealIActiveScriptSite::OnEnterScript(void)
{
	return(S_OK);
}

// Called immediately after the script engine executes/interprets each
// script.
STDMETHODIMP MyRealIActiveScriptSite::OnLeaveScript(void) 
{
	return(S_OK);
}





















//===========================================================================
//==================== IActiveScriptSiteWindow functions ====================
//===========================================================================

// Called by the script engine when it wants to know what window it should
// use as the owner of any dialog box the engine presents.
STDMETHODIMP MyRealIActiveScriptSite::GetWindow(HWND *phwnd)
{
	// Give it our main app window
	*phwnd = MainWindow;
	return(S_OK);
}

// Called when the script engine wants us to enable/disable all of our open
// windows.
STDMETHODIMP MyRealIActiveScriptSite::EnableModeless(BOOL enable)
{
	// We have only 1 open window -- our main window
	::EnableWindow(MainWindow, enable);
	return(S_OK);
}
