// Main functions for an example program that loads and executes a script.

#include <windows.h>
#include <initguid.h>
#include <commctrl.h>
#include <objbase.h>
#include <activscp.h>
#include <tchar.h>
#include "TraceWin.h"
#include "IActiveScriptSite.h"
#include "resource.h"
#include "extern.h"

// This app's handle
HINSTANCE			InstanceHandle;

// Main window of our app
HWND				MainWindow;

// A buffer to hold the name of the script to run
static TCHAR			Filename[MAX_PATH];

// The name of the function that is the entry point of any script we run
static const wchar_t	MainFunction[] = L"main";

// For script file dialog
static const TCHAR	Load[] = "Pick script to run (or Cancel to quit):";
static const TCHAR	ExtensionList[] = {'V','B','s','c','r','i','p','t',' ','(','*','.','v','b','s',')',0,
								'*','.','v','b','s',0,
								'J','S','c','r','i','p','t',' ','(','*','.','v','j','s',')',0,
								'*','.','j','v','s',0,
								'A','l','l',' ','f','i','l','e','s',' ','(','*','.','*',')',0,
								'*','.','*',0,0};

// For getting engine's GUID from registry
static const TCHAR		CLSIDStr[] = _T("CLSID");
static const TCHAR		ScriptEngineStr[] = _T("ScriptEngine");

// Error message box
static const TCHAR	ErrorStr[] = _T("Error");






/************************* getLoadName() ***********************
 * Gets the user's choice of script filename, and copies it
 * to Filename[].
 *
 * RETURNS: 0 if success, or non-zero if cancel/failure.
 *
 * NOTE: Filename[] must be MAX_PATH size.
 */

static BOOL getLoadName(void)
{
	OPENFILENAME	ofn;

	// Clear out fields
	ZeroMemory(&ofn, sizeof(OPENFILENAME));

	// Store passed buffer
	ofn.lpstrFile = &Filename[0];
	Filename[0] = 0;
	ofn.nMaxFile = MAX_PATH;

	// Set size
	ofn.lStructSize = sizeof(OPENFILENAME);

	// Set extensions
	ofn.lpstrFilter = &ExtensionList[0];

	// Set title
	ofn.lpstrTitle = (LPTSTR)&Load[0];

	// Set flags
	ofn.Flags = OFN_PATHMUSTEXIST|OFN_FILEMUSTEXIST|OFN_LONGNAMES|OFN_EXPLORER|OFN_HIDEREADONLY;

	// Present the dialog and get user's selection
	if (GetOpenFileName(&ofn))
	{
		// Return OK
		return(0);
	}

	// Abort
	return(1);
}





/********************* display_sys_error() ********************
 * Displays a messagebox for the passed OS error number.
 *
 * NOTE: If passed a 0, this calls GetLastError().
 */

void display_sys_error(DWORD err)
{
	TCHAR	buffer[160];

	if (!err) err = GetLastError();		// If passed a 0, caller wants us to call GetLastError(). Do it FIRST!
	buffer[0] = 0;
	FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, 0, err, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), &buffer[0], 160, 0);
	MessageBox(MainWindow, &buffer[0], &ErrorStr[0], MB_OK);
}





/*********************** loadUnicodeScript() ********************
 * Reads a script off of disk, and copies it to a UNICODE
 * buffer.
 *
 * fn =		Filename of script.
 *
 * RETURNS: A pointer to the allocated UNICODE buffer if success,
 * or zero if failure.
 *
 * NOTE: Caller must GlobalFree() the returned buffer.
 *
 * Displays an error message if a failure.
 */

static OLECHAR * loadUnicodeScript(LPCTSTR fn)
{
	register OLECHAR	*script;
	register HANDLE		hfile;

	// Assume an error
	script = 0;

	// Open the file
	if ((hfile = CreateFile(fn, GENERIC_READ, FILE_SHARE_READ, 0, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0)) != INVALID_HANDLE_VALUE)
	{
		DWORD	filesize;
		char	*psz;

		// Get a buffer to read in the file
		filesize = GetFileSize(hfile, 0);
		if ((psz = (char *)GlobalAlloc(GMEM_FIXED, filesize + 1)))
 		{
			DWORD	read;

			// Read in the file
			ReadFile(hfile, psz, filesize, &read, 0);

			// Get a buffer to convert to UNICODE (plus an extra wchar_t to nul-terminate it since
			// the engine's IActiveScriptParse ParseScriptText expects that)
			if ((script = (OLECHAR *)GlobalAlloc(GMEM_FIXED, (filesize + 1) * sizeof(OLECHAR))))
			{
				// Convert to UNICODE and nul-terminate
				MultiByteToWideChar(CP_ACP, 0, psz, filesize, script, filesize + 1);
				script[filesize] = 0;
			}
			else
				display_sys_error(0);

			GlobalFree(psz);
		}
		else
			display_sys_error(0);

		CloseHandle(hfile);
	}
	else
		display_sys_error(0);

	return(script);
}





/********************** display_COM_error() ********************
 * Displays a messagebox for a COM error.
 *
 * msg =		Format string for sprintf().
 * hr =			COM error number.
 *
 * NOTE: Total size of error msg must be < 256 TCHARs.
 */

void display_COM_error(LPCTSTR msg, HRESULT hr)
{
	TCHAR			buffer[256];

	wsprintf(&buffer[0], msg, hr);
	MessageBox(MainWindow, &buffer[0], &ErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
}





/************************** runScript() ***********************
 * Runs a script (on disk).
 *
 * fn =		Filename of script.
 * guid =	GUID of the script engine that runs the script.
 */

void runScript(LPCTSTR fn, GUID *guid)
{
	register HRESULT	hr;
	IActiveScriptParse	*activeScriptParse;
	IActiveScript		*activeScript;

	// Create an instance of the script engine, and get its IActiveScript object
	if ((hr = CoCreateInstance(guid, 0, CLSCTX_ALL, &IID_IActiveScript, (void **)&activeScript)))
		display_COM_error("Can't get engine's IActiveScript: %08X", hr);
	else
	{
		// Get the script engine's IActiveScriptParse object (which we can do from its
		// IActiveScript's QueryInterface since IActiveScriptParse is a
		// sub-object of the IActiveScript)
		if ((hr = activeScript->lpVtbl->QueryInterface(activeScript, &IID_IActiveScriptParse, (void **)&activeScriptParse)))
			display_COM_error("Can't get engine's IActiveScriptParse: %08X", hr);
		else
		{
			// Initialize the engine. This just lets the engine internally
			// initialize some stuff in preparation of us adding scripts to it
			// for the first time
			if ((hr = activeScriptParse->lpVtbl->InitNew(activeScriptParse)))
				display_COM_error("Can't initialize engine : %08X", hr);
			else
			{
				// Give the engine our IActiveScriptSite object. If all goes well,
				// the engine will call its QueryInterface (which will AddRef it)
				if ((hr = activeScript->lpVtbl->SetScriptSite(activeScript, (IActiveScriptSite *)&MyActiveScriptSite)))
					display_COM_error("Can't set our IScriptSite : %08X", hr);
				else
				{
					register LPOLESTR	str;

					// Load the script from disk. NOTE: We need to load it UNICODE for ParseScriptText()
					if (!(str = loadUnicodeScript(fn)))
						MessageBox(MainWindow, "Can't load script from disk", &ErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
					else
					{
						// Have the script engine parse it and add it to its internal list
						// of scripts to run. NOTE: We don't pass any object name, so this
						// script is put inside the "global or default object" and therefore
						// this script will be run as soon as we set the engine to the
						// connected state
						hr = activeScriptParse->lpVtbl->ParseScriptText(activeScriptParse, str, 0, 0, 0, 0, 0, 0, 0, 0);

						// We no longer need the loaded script
						GlobalFree(str);

						// NOTE: If the script engine has a problem parsing/tokenizing the script, it will
						// have called our IActiveScriptSite's OnScriptError() to display an error msg, so
						// we don't need to do that here
						if (!hr &&

							// Set engine's state to CONNECTED. NOTE: If we called the engine's AddNamedItem()
							// to add some objects, then the script engine will QueryInterface our
							// IActiveScriptSite for the needed IDispatch objects from us
							(hr = activeScript->lpVtbl->SetScriptState(activeScript, SCRIPTSTATE_CONNECTED)))
						{
							display_COM_error("Engine can't connect events: %08X", hr);
						}
					}
				}
			}

			// Release script engine's IActiveScriptParse
			activeScriptParse->lpVtbl->Release(activeScriptParse);
		}

		// We're supposed to Close() the engine before we Release() the IActiveScript
		activeScript->lpVtbl->Close(activeScript);
		
		// Release script engine's IActiveScript
		activeScript->lpVtbl->Release(activeScript);
	}
}





/**************************** mainWndProc() *****************************
 * The message procedure for MainWindow. It is called by Windows whenever
 * there is a message for MainWindow to process.
 */

long APIENTRY mainWndProc(HWND hwnd, UINT uMsg, UINT wParam, long lParam)
{
	switch(uMsg)
	{
		// ***************************************************************
		// ======================== Menu or Buttons ======================
		case WM_COMMAND:
		{
			switch (wParam)
			{
				// -------------------- MENU: File -> Run script
		        case IDM_FILE_RUNSCRIPT:
				{
					// Get the filename of the script to run
					if (!getLoadName())
					{
						LPCTSTR		extension;
						wchar_t		buffer[100];
						HKEY		hk;

						// Ok, let's see if we can find the appropriate script engine to use,
						// based upon the extension on the filename. Hopefully the script
						// engine associated itself with a particular file extension. If so,
						// then it should have set a registry key under HKEY_CLASSES_ROOT.

						// Isolate the extension on the filename
						extension = &Filename[0] + lstrlen(&Filename[0]);
						goto backup;
						while (*extension != '.')
						{
							if (*extension == '\\' || extension <= &Filename[0])
							{
								// No extension. We'll need to let the user pick out the engine
								goto choose;
							}

backup:						extension = CharPrev(&Filename[0], extension);
						}

						// See if the engine set a file association
						if (RegOpenKeyEx(HKEY_CLASSES_ROOT, extension, 0, KEY_QUERY_VALUE|KEY_READ, &hk))
						{
							// It didn't. We'll have to let the user pick a script engine, and
							// copy the GUID to &buffer[60]
choose:						lParam = chooseEngineDlg((GUID *)&buffer[60]);
						}
						else
						{
							HKEY	subKey;

							wParam = REG_SZ;
							lParam = sizeof(buffer);
							lParam = RegQueryValueEx(hk, 0, 0, &wParam, (LPBYTE)&buffer[0], &lParam);
							RegCloseKey(hk);
							if (lParam) goto choose;

							// The engine set an association. We got the Language string in buffer[]. Now
							// we can use it to look up the engine's GUID

							// Open HKEY_CLASSES_ROOT\{LanguageName}
again:						lParam = sizeof(buffer);
							if (!RegOpenKeyEx(HKEY_CLASSES_ROOT, (LPCTSTR)&buffer[0], 0, KEY_QUERY_VALUE|KEY_READ, &hk))
							{
								// Read the GUID (in string format) into buffer[] by querying the value of CLSID
								if (!RegOpenKeyEx(hk, &CLSIDStr[0], 0, KEY_QUERY_VALUE|KEY_READ, &subKey))
								{
									lParam = RegQueryValueExW(subKey, 0, 0, &wParam, (LPBYTE)&buffer[0], &lParam);
									RegCloseKey(subKey);
								}
								else if (extension)
								{
									// If an error, see if we have a "ScriptEngine" key under here that contains
									// the real language name
									if (!RegOpenKeyEx(hk, &ScriptEngineStr[0], 0, KEY_QUERY_VALUE|KEY_READ, &subKey))
									{
										lParam = RegQueryValueEx(subKey, 0, 0, &wParam, (LPBYTE)&buffer[0], &lParam);
										RegCloseKey(subKey);
										if (!lParam)
										{
											RegCloseKey(hk);
											extension = 0;
											goto again;
										}
									}
								}

								RegCloseKey(hk);

								// Convert the GUID string to a GUID and put it in &buffer[60]
								if (!lParam && (lParam = CLSIDFromString(&buffer[0], (GUID *)&buffer[60])))
									display_COM_error("Can't get engine GUID: %08X", lParam);
							}
						}

						// Now that we've got the GUID, and the script filename, run the script
						if (!lParam) runScript(&Filename[0], (GUID *)&buffer[60]);
					}

					break;	// End of IDM_FILE_RUNSCRIPT
				}
			}

			break;
		}

		// ******************************************************************
		case WM_SETFOCUS:
		{
			SetFocus(GetDlgItem(hwnd, IDC_TRACE));
			break;
		}

		// ******************************************************************
		case WM_SIZE:
		{
			MoveWindow(GetDlgItem(hwnd, IDC_TRACE), 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);
			break;
		}

		// ******************************************************************
		// ====================== Create main window ========================
		case WM_INITDIALOG:
		{
			HICON	icon;

			// Load/set icon for System menu on the window. I put my icon
			// in this executable's resources. Note that windows frees this
			// when my window is closed
			if ((icon = LoadIcon(InstanceHandle,MAKEINTRESOURCE(IDI_MAIN_ICON))))
				SetClassLong(hwnd, GCL_HICON, (LONG)icon);
  
			SendDlgItemMessage(hwnd, IDC_TRACE, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), 0);

			return(1);
		}

		// ******************************************************************
		// =================== User wants to close window ===================
		case WM_CLOSE:
		{
			// Close this window
			DestroyWindow(hwnd);

			return(1);
		}

		case WM_DESTROY:
		{
 			// Post the WM_QUIT message to quit the message loop in WinMain()
			PostQuitMessage(0);

			return(1);
		}
	}

	// Indicate that I didn't handle the msg
	return(0);
} 





/************************** WinMain() **************************
 * Our EXE's entry point. This is called by Windows when our
 * EXE first starts up.
 */

int APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	HRESULT				hr;

	// Save Instance handle which I need when opening the "Choose engine" dialog
	// box. That dialog is linked into this executable's resources, so I need to pass
	// this handle to DialogBox() when I open that dialog
	InstanceHandle = hInstance;

	// Loads Windows common control's DLL
	InitCommonControls();

	// MainWindow not open yet
	MainWindow = 0;

	// Initialize COM DLL
	if ((hr = CoInitialize(0)))
		display_COM_error("Failed to initialize COM: %08X", hr);
	else
	{
		// Initialize MyRealIActiveScriptSite object
		initIActiveScriptSiteObject();

		// Create Main window
		if (!(MainWindow = CreateDialog(InstanceHandle, MAKEINTRESOURCE(IDD_MAINWINDOW), 0, mainWndProc)))
			display_sys_error(0);
		else
		{
			MSG		msg;

			// Show the window with default size/position
			ShowWindow(MainWindow, SW_SHOWDEFAULT);
			UpdateWindow(MainWindow);

			// Here's our message loop which is executed until the user closes
			// down our MainWindow
			while (GetMessage(&msg, 0, 0, 0) == 1)
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}

		// Allow our IActiveScriptSite to free any resource
		MyActiveScriptSite.site.lpVtbl->Release((IActiveScriptSite *)&MyActiveScriptSite);
	}

	// Free COM
	CoUninitialize();

	return(0);
}
