// This is a C example that tests the ISort COM component (in ISort.dll).
//
// ISort.DLL has a COM component known as an ISort. This component
// also has an IConnectionPointContainer sub-object. So ISort supports us
// giving it some object that wraps some of our callback functions. Specifically,
// ISort.h defines an ICompare object we can create to wrap our callback
// functions. ICompare contains one callback functions we can write. ICompare's
// VTable uses the name "Compare" for that function's pointer in the VTable.
// We'll use the same name for the function itself.

#include <windows.h>
#include <objbase.h>
#include <activscp.h>
#include <stdio.h>
#include "../ISort/ISort.h"

// An array of 5 DWORDs to be sorted
DWORD Array[5] = {2, 3, 1, 5, 4};


// ISort supports us giving it only 1 ICompare object, so for simplicity, we
// can just declare it as a global. That way, we don't need to allocate it, free it,
// nor maintain a reference count for it. Here's our 1 ICompare object.
static ICompare		MyCompare;


static HRESULT STDMETHODCALLTYPE QueryInterface(ICompare *this, REFIID vTableGuid, void **ppv)
{
	// This is an ICompare object, so we must recognize ICompare VTable's GUID,
	// which is defined for us in ISort.h. Our ICompare can also masquerade
	// as an IUnknown
	if (!IsEqualIID(vTableGuid, &IID_IUnknown) && !IsEqualIID(vTableGuid, &DIID_ICompare))
	{
		*ppv = 0;
		return(E_NOINTERFACE);
	}

	*ppv = this;

	// Normally, we'd call our AddRef function here. But since our ICompare isn't
	// allocated, we don't need to bother with reference counting
	return(NOERROR);
}

static ULONG STDMETHODCALLTYPE AddRef(ICompare *this)
{
	// Our one and only ICompare isn't allocated. Instead, it is statically
	// declared above (MyCompare). So we'll just return a 1
	return(1);
}

static ULONG STDMETHODCALLTYPE Release(ICompare *this)
{
	return(1);
}

// This is the extra function for ICompare. It is called
// Compare. The ISort object will call this function when
// we call ISort's Sort function.

static long STDMETHODCALLTYPE Compare(ICompare *this, const void *elem1, const void *elem2)
{
	// Do a compare of the two elements. We happen to know that
	// we'll be passing an array of DWORD values to Sort()
	if (*((DWORD *)elem1) == *((DWORD *)elem2)) return 0;
	if (*((DWORD *)elem1) < *((DWORD *)elem2)) return -1;
	return 1;
}


// Our ICompare VTable. We need only one of these, so we can
// declare it static
static const ICompareVtbl ICompare_Vtbl = {QueryInterface,
AddRef,
Release,
Compare};




int main(int argc, char **argv)
{
	ISort						*sortObj;
	IConnectionPointContainer	*container;
	IConnectionPoint			*point;
	HRESULT						hr;
	DWORD						cookie;

	// Although we don't have to allocate our ICompare, we do need to
	// initialize it, by storing its VTable in its lpVtbl member
	MyCompare.lpVtbl = (ICompareVtbl *)&ICompare_Vtbl;

	if (!CoInitialize(0))
	{
		// Get an ISort object
		if ((hr = CoCreateInstance(&CLSID_ISort, 0, CLSCTX_ALL, &IID_ISort, &sortObj)))
			MessageBox(0, "Can't create ISort object", "CreateInstance error", MB_OK|MB_ICONEXCLAMATION);
		else
		{
			// Get ISort' IConnectionPointContainer sub-object. We do this by calling
			// ISort' QueryInterface, and pass it the standard GUID for an
			// IConnectionPointContainer VTable
			if ((hr = sortObj->lpVtbl->QueryInterface(sortObj, &IID_IConnectionPointContainer, &container)))
				MessageBox(0, "Can't get IConnectionPointContainer object", "QueryInterface error", MB_OK|MB_ICONEXCLAMATION);
			else
			{
				// Get ISort' IConnectionPoint sub-object for specifically giving
				// ISort our ICompare. We do this by calling IConnectionPointContainer's
				// FindConnectionPoint, and pass it ICompare VTable's GUID (defined for us in
				// ISort.h)
				hr = container->lpVtbl->FindConnectionPoint(container, &DIID_ICompare, &point);

				// We don't need the IConnectionPointContainer, now that we got the one IConnectionPoint
				// we want (ie, the one we use to give ISort our ICompare)
				container->lpVtbl->Release(container);

				if (hr)
					MessageBox(0, "Can't get IConnectionPoint object", "FindConnectionPoint error", MB_OK|MB_ICONEXCLAMATION);
				else
				{
					// Now call the IConnectionPoint's Advise function, giving it some object
					// whose QueryInterface it will call to get our ICompare object. Let's
					// just give it our ICompare object. So Advise will call our ICompare's
					// QueryInterface to tell us to give it a pointer to our ICompare. Dumb?
					// You bet. All of this convoluted stuff is a combination of poor pre-planning
					// by Microsoft programmers when they designed this stuff, as well as the
					// colossal blunder of designing COM to accomodate the limitations of early,
					// primitive editions of Visual Basic.
					//
					// Advise() gives us back a "cookie" value that we'll need to later pass to
					// Unadvise() (when we want to tell ISort to stop using our ICompare)
					if ((hr = point->lpVtbl->Advise(point, (IUnknown *)&MyCompare, &cookie)))
						MessageBox(0, "Can't set our ICompare object", "Advise error", MB_OK|MB_ICONEXCLAMATION);
					else
					{
						// Call ISort' Sort() function to sort our array of DWORDs.
						sortObj->lpVtbl->Sort(sortObj, &Array[0], 5, sizeof(DWORD));

						// Tell ISort to stop using our ICompare (and Release it). We pass
						// the cookie value we got from Advise() above
						if ((hr = point->lpVtbl->Unadvise(point, cookie)))
							MessageBox(0, "Can't unset our ICompare object", "Unadvise error", MB_OK|MB_ICONEXCLAMATION);

						// Release the IConnectionPoint now that we're done with it (and
						// we've told ISort to stop calling our ICompare functions)
						point->lpVtbl->Release(point);
					}
				}

			}

			// Release the ISort now that we're done with it
			sortObj->lpVtbl->Release(sortObj);
		}

		// When finally done with OLE, free it
		CoUninitialize();
	}
	else
		MessageBox(0, "Can't initialize COM", "CoInitialize error", MB_OK|MB_ICONEXCLAMATION);

	return(0);
}
