// C source code to a simple COM object, compiled into an ordinary
// dynamic link library (DLL). This demonstrates adding
// IConnectionPointContainer and IConnectionPoint sub-objects to
// our ISort object for the purpose of letting an app
// supply some callback functions (for ISort's extra functions
// to call)

#include <windows.h>
#include <objbase.h>
#include <activscp.h>
#include <olectl.h>
#include <stddef.h>
#include "ISort.h"






static DWORD		OutstandingObjects;
static DWORD		LockCount;









// Our ISort object ////////////////////////////////////////////////////////////

// This is our ISort object. It can "source events" which just
// means that we allow an app to give us some object that has callback
// functions that ISort's functions can call for whatever purpose
// we deem. We define a ICompare object that an app can give us
// (containing the callback functions). This ICompare's extra functions
// must be exactly as we've defined in ISort.h.
//
// Because we need to add some extra, private members to our ISort
// object, we'll define a MyRealISort object below, with those extra
// members.
//
// Because our ISort sources events, it must have an
// IConnectionPointContainer sub-object. (ie, Our MyRealISort
// has multiple interfaces). The app uses our IConnectionPointContainer
// sub-object to get a IConnectionPoint sub-object that allows him to
// give us his ICompare object. To make this easy, we'll just embed
// our IConnectionPointContainer sub-object right inside of our
// MyRealISort.
//
// We also embed an IConnectionPoint sub-object. We're going to
// support allowing the app only one ICompare object per each of
// our ISort objects.
//
// Because we support only one ICompare per ISort, we'll
// store a pointer to the app's ICompare in our ISort.
typedef struct {
	ISortVtbl					*lpVtbl;		// ISort's VTable
	DWORD						count;			// ISort's reference count
	IConnectionPointContainer	container;		// ISort's IConnectionPointContainer sub-object
	IConnectionPoint			point;			// ISort's IConnectionPoint sub-object
	ICompare					*compare;		// Pointer to an app's ICompare
} MyRealISort;

static HRESULT STDMETHODCALLTYPE QueryInterface(ISort *this, REFIID vTableGuid, void **ppv)
{
	// Because our ISort sources events, we must return an IConnectionPointContainer
	// sub-object if the app asks for one. Because we've embedded our
	// IConnectionPointContainer object inside of our MyRealISort, we can get that
	// sub-object very easily using pointer arithmetic
	if (IsEqualIID(vTableGuid, &IID_IConnectionPointContainer))
		*ppv = ((unsigned char *)this + offsetof(MyRealISort, container));

	else if (IsEqualIID(vTableGuid, &IID_IUnknown) || IsEqualIID(vTableGuid, &IID_ISort))
		*ppv = this;

	else
	{
		*ppv = 0;
		return(E_NOINTERFACE);
	}

	this->lpVtbl->AddRef(this);

	return(NOERROR);
}

static ULONG STDMETHODCALLTYPE AddRef(ISort *this)
{
	return(++((MyRealISort *)this)->count);
}

static ULONG STDMETHODCALLTYPE Release(ISort *this)
{
	// Note: This count includes any outstanding IConnectionPoint
	// and IConnectionPointContainer objects that the app is still
	// holding onto. So we don't actually free our ISort
	// until all of those sub-objects are released too
	if (--((MyRealISort *)this)->count == 0)
	{
		GlobalFree(this);
		InterlockedDecrement(&OutstandingObjects);
		return(0);
	}
	return(((MyRealISort *)this)->count);
}

// Our extra function for ISort. Sort() takes array of items and
// sorts it. It calls the app ICompare's Compare function to compare
// two items.
static HRESULT STDMETHODCALLTYPE Sort(ISort *this, void *base, DWORD numElems, DWORD sizeElem)
{
	register ICompare	*compare;

	// Get the app's ICompare object. We stored a pointer to it in
	// our MyRealISort->compare member. NOTE: If this member is 0,
	// then the app has not yet gotten our IConnectionPoint object,
	// and called its Advise() function to give us the app's ICompare.
	// Or perhaps, the app has called our IConnectionPoint->Unadvise() to
	// tell us to Release(), and no longer use, its ICompare
	if ((compare = ((MyRealISort *)this)->compare))
	{
		// Ok, the app has already given us its ICompare. We can do the sort
		void	*hi;
		void	*p;
		void	*lo;
		void	*tmp;

		if ((tmp = GlobalAlloc(GMEM_FIXED, sizeElem)))
		{
			hi = ((char *)base + ((numElems - 1) * sizeElem));
			lo = base;

			while (hi > base)
			{
				lo = base;
				p = ((char *)base + sizeElem);
				while (p <= hi)
				{
					if (compare->lpVtbl->Compare(compare, p, lo) > 0) lo = p;
					(char *)p += sizeElem;
				}

				CopyMemory(tmp, lo, sizeElem);
				CopyMemory(lo, hi, sizeElem);
				CopyMemory(hi, tmp, sizeElem);
				(char *)hi -= sizeElem;
			}

			GlobalFree(tmp);
			return(S_OK);
		}

		return(E_OUTOFMEMORY);
	}

	return(E_FAIL);
}

// Our ISort VTable. We need only 1 of these for all
// our ISort objects
static const ISortVtbl ISort_Vtbl = {QueryInterface,
AddRef,
Release,
Sort};




















// Our IConnectionPointContainer sub-object (for ISort) ////////////////////////

static STDMETHODIMP QueryInterface_Connect(IConnectionPointContainer *this, REFIID vTableGuid, void **ppv)
{
	// Because this is a sub-object of our ISort (ie, MyRealISort) object,
	// we delegate to ISort's QueryInterface. And because we embedded the
	// IConnectionPointContainer directly inside of MyRealISort, all we need
	// is a little pointer arithmetic to get our ISort
	return(QueryInterface((ISort *)((char *)this - offsetof(MyRealISort, container)), vTableGuid, ppv));
}

static STDMETHODIMP_(ULONG) AddRef_Connect(IConnectionPointContainer *this)
{
	// Because we're a sub-object of ISort, delegate to its AddRef()
	// in order to increment ISort's reference count
	return(AddRef((ISort *)((char *)this - offsetof(MyRealISort, container))));
}

static STDMETHODIMP_(ULONG) Release_Connect(IConnectionPointContainer *this)
{
	// Because we're a sub-object of ISort, delegate to its Release()
	// in order to decrement ISort's reference count
	return(Release((ISort *)((char *)this - offsetof(MyRealISort, container))));
}

static STDMETHODIMP EnumConnectionPoints(IConnectionPointContainer *this, IEnumConnectionPoints **enumPoints)
{
	// The app had better know the GUIDs of whatever objects our
	// ISort supports for callbacks (ie, an ICompare), because
	// we're not going to bother providing him with an object to
	// enumerate the VTable GUIDs of all those supported objects
	*enumPoints = 0;
	return(E_NOTIMPL);
}
 
static STDMETHODIMP FindConnectionPoint(IConnectionPointContainer *this, REFIID vTableGuid, IConnectionPoint **ppv) 
{
	// Is the app asking us to return an IConnectionPoint object it can use
	// to give us its ICompare object? The app asks this by passing us
	// ICompare VTable's GUID (which we defined in ISort.h)
	if (IsEqualIID(vTableGuid, &DIID_ICompare))
	{
		register MyRealISort		*iSort;

		// The app obviously wants to connect its ICompare object to our
		// ISort. In order to do that, we need to give the app a standard
		// IConnectionPoint, so the app can call its Advise function to
		// give us its ICompare. This is easy to do since we embedded both
		// our IConnectionPointContainer and IConnectionPoint inside of our 
		// ISort. All we need is a little pointer arithmetic
		iSort = (MyRealISort *)((char *)this - offsetof(MyRealISort, container));
		*ppv = &iSort->point;

		// Because we're giving the app a pointer to our IConnectionPoint,
		// and our IConnectionPoint is a sub-object of ISort, we need to
		// increment ISort's reference count. The easiest way to do this is
		// to call our IConnectionPointContainer's AddRef, because all we do
		// there is delegate to our ISort's AddRef
		AddRef_Connect(this);

		return(S_OK);
	}

	// We don't support any other app objects connecting to ISort
	// events. All we've defined, and support, is an ICompare object. Tell
	// the app we don't know anything about the GUID he passed to us, and
	// do not give him any IConnectionPoint object
	*ppv = 0;
	return(E_NOINTERFACE);
}

// Our IConnectionPointContainer VTable. We need only 1 of these for all
// our IConnectionPointContainer objects
static const IConnectionPointContainerVtbl IConnectionPointContainer_Vtbl = {QueryInterface_Connect,
AddRef_Connect,
Release_Connect,
EnumConnectionPoints,
FindConnectionPoint};













// Our IConnectionPoint sub-object (for ISort) ////////////////////////////

static STDMETHODIMP QueryInterface_Point(IConnectionPoint *this, REFIID vTableGuid, void **ppv)
{
	// Because this is a sub-object of our ISort (ie, MyRealISort) object,
	// we delegate to ISort's QueryInterface. And because we embedded the
	// IConnectionPoint directly inside of MyRealISort, all we need
	// is a little pointer arithmetic to get our ISort
	return(QueryInterface((ISort *)((char *)this - offsetof(MyRealISort, point)), vTableGuid, ppv));
}

static STDMETHODIMP_(ULONG) AddRef_Point(IConnectionPoint *this)
{
	// Because we're a sub-object of ISort, delegate to its AddRef()
	// in order to increment ISort's reference count
	return(AddRef((ISort *)((char *)this - offsetof(MyRealISort, point))));
}

static STDMETHODIMP_(ULONG) Release_Point(IConnectionPoint *this)
{
	// Because we're a sub-object of ISort, delegate to its Release()
	// in order to decrement ISort's reference count
	return(Release((ISort *)((char *)this - offsetof(MyRealISort, point))));
}

// Called by the app to get our ICompare VTable's GUID (which we defined in ISort.h).
// The app would call GetConnectionInterface() if it didn't link with ISort.h, and
// therefore doesn't know our ICompare VTable's GUID. The app needs to know this GUID
// because our Advise function below is going to pass this same GUID to some app object's
// QueryInterface. The app's QueryInterface had better recognize this GUID if it intends
// to honor our request to give us its ICompare object
static STDMETHODIMP GetConnectionInterface(IConnectionPoint *this, IID *vTableGuid) 
{
	// Tell the app to recognize our ICompare VTable GUID (defined as
	// DIID_ICompare in ISort.h) when our Advise function calls
	// some app QueryInterface function
	CopyMemory(vTableGuid, &DIID_ICompare, sizeof(GUID));
	return(S_OK);
}
 
// Called by the app to get the IConnectionPointContainer sub-object for our
// ISort object.
static STDMETHODIMP GetConnectionPointContainer(IConnectionPoint *this, IConnectionPointContainer **ppv) 
{
	register MyRealISort	*iSort;

	// Get the MyRealISort that this IConnectionPoint sub-object belongs to.
	// Because this IConnectPoint sub-object is embedded directly inside its
	// MyRealISort, all we need is a little pointer arithmetic
	iSort = (MyRealISort *)((char *)this - offsetof(MyRealISort, point));

	// Because the IConnectionPointContainer sub-object is also embedded right
	// inside the same MyRealISort, we can get a pointer to it easily as so
	*ppv = &iSort->container;

	// Because we're giving the app a pointer to our IConnectionPointContainer,
	// and our IConnectionPointContainer is a sub-object of ISort, we need to
	// increment ISort's reference count. The easiest way to do this is to call
	// our IConnectionPoint's AddRef, because all we do there is delegate
	// to our ISort's AddRef
	AddRef_Point(this);

	return(S_OK);
}

// Called by the app to give us its ICompare object. Actually, the app doesn't
// just give us its ICompare. Rather, the app calls our Advise, passing us some
// app object from which we can request the app to give us its ICompare. All of
// this convoluted stuff is a combination of poor pre-planning by Microsoft
// programmers when they designed this stuff, as well as the colossal blunder of
// designing COM to accomodate the limitations of early, primitive editions of
// Visual Basic.
//
// The second arg passed here is some app object whose QueryInterface function
// we call to request the app's ICompare. We pass the GUID DIID_ICompare to
// this QueryInterface in order to tell the app to give us its ICompare
static STDMETHODIMP Advise(IConnectionPoint *this, IUnknown *obj, DWORD *cookie) 
{
	register HRESULT		hr;
	register MyRealISort	*iSort;

	// Get the MyRealISort that this IConnectionPoint sub-object belongs
	// to. Because this IConnectPoint sub-object is embedded directly
	// inside its MyRealISort, all we need is a little pointer arithmetic
	iSort = (MyRealISort *)((char *)this - offsetof(MyRealISort, point));

	// We allow only one ICompare for our ISort, so see if the app already
	// called our Advise(), and we got one. If so, let the app know that it
	// is trying to give us more ICompares than we allow
	if (iSort->compare) return(CONNECT_E_ADVISELIMIT);
 
	// Ok, we haven't yet gotten the one ICompare we allow from the app. Get
	// the app's ICompare object. We do this by calling the QueryInterface
	// function of the app object passed to us. We pass ICompare VTable's
	// GUID (which we defined in ISort.h).
	//
	// Save the app's ICompare pointer in our ISort  compare member, so we
	// can get it when we need it
	hr = obj->lpVtbl->QueryInterface(obj, &DIID_ICompare, (void **)&iSort->compare);

	// We need to return (to the app) some value that will clue our Unadvise()
	// function below how to locate this app ICompare. The simpliest thing is
	// to just use the app's ICompare pointer as that return value
	*cookie = (DWORD)iSort->compare;

	return(hr);
}

// Called by the app to tell us to stop using, and Release(), its ICompare object.
// The second arg passed here is the value our Advise() function above returned when
// we got the ICompare from the app. This value should help us locate wherever we
// stored that ICompare pointer we got in Advise()
static STDMETHODIMP Unadvise(IConnectionPoint *this, DWORD cookie) 
{
	register MyRealISort	*iSort;

	// Get the MyRealISort that this IConnectionPoint sub-object belongs
	// to. Because this IConnectPoint sub-object is embedded directly
	// inside its MyRealISort, all we need is a little pointer arithmetic
	iSort = (MyRealISort *)((char *)this - offsetof(MyRealISort, point));

	// Use the passed value to find wherever we stored his ICompare pointer.
	// Well, since we allow only one ICompare for our ISort, we already
	// know we stored it in our ISort->compare member. And Advise()
	// returned that pointer as the "cookie" value. So we already got the
	// ICompare right now.
	//		
	// Let's just make sure the cookie he passed is really the pointer we expect
	if (cookie && (ICompare *)cookie == iSort->compare)
	{
		// Release the app's ICompare
		((ICompare *)cookie)->lpVtbl->Release((ICompare *)cookie);

		// We no longer have the app's ICompare, so clear the ISort
		// compare member
		iSort->compare = 0;

		return(S_OK);
	}
	return(CONNECT_E_NOCONNECTION);
}

// Called by the app get our IEnumConnections object. The app uses this object
// to discover all of the ICompare objects that have been given to us. But since
// we support only 1 ICompare given to us, we won't provide him an IEnumConnections
// object. The hell with it
static STDMETHODIMP EnumConnections(IConnectionPoint *this, IEnumConnections **enumConnects)
{
	*enumConnects = 0; 
	return(E_NOTIMPL);
}

// Our IConnectionPoint VTable. We need only 1 of these for all
// our IConnectionPoint objects
static const IConnectionPointVtbl IConnectionPoint_Vtbl = {
QueryInterface_Point,
AddRef_Point,
Release_Point,
GetConnectionInterface,
GetConnectionPointContainer,
Advise,
Unadvise,
EnumConnections};

















// Our IClassFactory object ///////////////////////////////////////////////////////

static IClassFactory	MyIClassFactoryObj;

static ULONG STDMETHODCALLTYPE classAddRef(IClassFactory *this)
{
	InterlockedIncrement(&OutstandingObjects);
	return(1);
}

static HRESULT STDMETHODCALLTYPE classQueryInterface(IClassFactory *this, REFIID factoryGuid, void **ppv)
{
	if (IsEqualIID(factoryGuid, &IID_IUnknown) || IsEqualIID(factoryGuid, &IID_IClassFactory))
	{
		this->lpVtbl->AddRef(this);

		*ppv = this;

		return(NOERROR);
	}

	*ppv = 0;
	return(E_NOINTERFACE);
}

static ULONG STDMETHODCALLTYPE classRelease(IClassFactory *this)
{
	return(InterlockedDecrement(&OutstandingObjects));
}

static HRESULT STDMETHODCALLTYPE classCreateInstance(IClassFactory *this, IUnknown *punkOuter, REFIID vTableGuid, void **objHandle)
{
	HRESULT						hr;
	register MyRealISort	*thisobj;

	*objHandle = 0;

	if (punkOuter)
		hr = CLASS_E_NOAGGREGATION;
	else
	{
		if (!(thisobj = (MyRealISort *)GlobalAlloc(GMEM_FIXED, sizeof(MyRealISort))))
			hr = E_OUTOFMEMORY;
		else
		{
			thisobj->lpVtbl = (ISortVtbl *)&ISort_Vtbl;
			thisobj->count = 1;

			// Our MyRealISort is a multiple interface object. It has an
			// IConnectionPointContainer sub-object embedded directly inside of
			// it. And we just allocated it when we allocated the MyRealISort
			// above. Now we need to set its VTable into its lpVtbl member and
			// we're done initializing this sub-object
			thisobj->container.lpVtbl = (IConnectionPointContainerVtbl *)&IConnectionPointContainer_Vtbl;

			// Our MyRealISort also has an IConnectionPoint sub-object
			// embedded directly inside of it. And we just allocated it when we
			// allocated the MyRealISort above. Now we need to set its
			// VTable into its lpVtbl member and we're done initializing this
			// sub-object
			thisobj->point.lpVtbl = (IConnectionPointVtbl *)&IConnectionPoint_Vtbl;

			// The app hasn't given us its ICompare object yet
			thisobj->compare = 0;

			hr = ISort_Vtbl.QueryInterface((ISort *)thisobj, vTableGuid, objHandle);
			ISort_Vtbl.Release((ISort *)thisobj);
			if (!hr) InterlockedIncrement(&OutstandingObjects);
		}
	}

	return(hr);
}

static HRESULT STDMETHODCALLTYPE classLockServer(IClassFactory *this, BOOL flock)
{
	if (flock) InterlockedIncrement(&LockCount);
	else InterlockedDecrement(&LockCount);

	return(NOERROR);
}

static const IClassFactoryVtbl IClassFactory_Vtbl = {classQueryInterface,
classAddRef,
classRelease,
classCreateInstance,
classLockServer};














// Miscellaneous functions ///////////////////////////////////////////////////////

HRESULT PASCAL DllGetClassObject(REFCLSID objGuid, REFIID factoryGuid, void **factoryHandle)
{
	register HRESULT		hr;

	if (IsEqualCLSID(objGuid, &CLSID_ISort))
		hr = classQueryInterface(&MyIClassFactoryObj, factoryGuid, factoryHandle);
	else
	{
		*factoryHandle = 0;
		hr = CLASS_E_CLASSNOTAVAILABLE;
	}

	return(hr);
}

HRESULT PASCAL DllCanUnloadNow(void)
{
	return((OutstandingObjects | LockCount) ? S_FALSE : S_OK);
}

BOOL WINAPI DllMain(HINSTANCE instance, DWORD fdwReason, LPVOID lpvReserved)
{
	switch (fdwReason)
	{
		case DLL_PROCESS_ATTACH:
		{
			OutstandingObjects = LockCount = 0;

			MyIClassFactoryObj.lpVtbl = (IClassFactoryVtbl *)&IClassFactory_Vtbl;

			DisableThreadLibraryCalls(instance);
		}
	}

	return(1);
}
