// This is a C example that tests the IExampleEvts COM component (in IExampleEvts.dll).
//
// IExampleEvts.DLL has a COM component known as an IExampleEvts. This component
// also has an IConnectionPointContainer sub-object. So IExampleEvts supports us
// giving it some object that wraps some of our callback functions. Specifically,
// IExampleEvts.h defines an IFeedback object we can create to wrap our callback
// functions. IFeedback contains five callback functions we can write. IFeedback's
// VTable uses the names Callback1, Callback2, Callback3, Callback4, and Callback5
// for those lpVtbl pointers. We'll use the same names for the functions themselves.

#include <windows.h>
#include <objbase.h>
#include <activscp.h>
//#include <olectl.h>
#include <stdio.h>
#include "../IExampleEvts/IExampleEvts.h"

// IExampleEvts supports us giving it only 1 IFeedback object, so for simplicity, we
// can just declare it as a global. That way, we don't need to allocate it, free it,
// nor maintain a reference count for it. Here's our 1 IFeedback object.
static IFeedback		MyFeedback;


static HRESULT STDMETHODCALLTYPE QueryInterface(IFeedback *this, REFIID vTableGuid, void **ppv)
{
	// This is an IFeedback object, so we must recognize IFeedback VTable's GUID,
	// which is defined for us in IExampleEvts.h. Our IFeedback can also masquerade
	// as an IUnknown
	if (!IsEqualIID(vTableGuid, &IID_IUnknown) && !IsEqualIID(vTableGuid, &DIID_IFeedback))
	{
		*ppv = 0;
		return(E_NOINTERFACE);
	}

	*ppv = this;

	// Normally, we'd call our AddRef function here. But since our IFeedback isn't
	// allocated, we don't need to bother with reference counting
	return(NOERROR);
}

static ULONG STDMETHODCALLTYPE AddRef(IFeedback *this)
{
	// Our one and only IFeedback isn't allocated. Instead, it is statically
	// declared above (MyFeedback). So we'll just return a 1
	return(1);
}

static ULONG STDMETHODCALLTYPE Release(IFeedback *this)
{
	return(1);
}

// This is the first extra function for IFeedback. It is called
// Callback1. The IExampleEvts object may call this function when
// we call one of IExampleEvts functions (such as DoSomething).

static HRESULT STDMETHODCALLTYPE Callback1(IFeedback *this)
{
	// Let's just display a message that Callback1 has been called
	// by one of IExampleEvts' functions.
	printf("My IFeedback's Callback1 function is called.\n");

	return(NOERROR);
}

// This is the second extra function for IFeedback. It is called
// Callback2. The IExampleEvts object may call this function when
// we call one of IExampleEvts functions (such as DoSomething).

static HRESULT STDMETHODCALLTYPE Callback2(IFeedback *this)
{
	// Let's just display a message that Callback2 has been called
	// by one of IExampleEvts' functions.
	printf("My IFeedback's Callback2 function is called.\n");

	return(NOERROR);
}

// Etc.

static HRESULT STDMETHODCALLTYPE Callback3(IFeedback *this)
{
	printf("My IFeedback's Callback3 function is called.\n");
	return(NOERROR);
}

static HRESULT STDMETHODCALLTYPE Callback4(IFeedback *this)
{
	printf("My IFeedback's Callback4 function is called.\n");
	return(NOERROR);
}

static HRESULT STDMETHODCALLTYPE Callback5(IFeedback *this)
{
	printf("My IFeedback's Callback5 function is called.\n");
	return(NOERROR);
}

// Our IFeedback VTable. We need only one of these, so we can
// declare it static
static const IFeedbackVtbl IFeedback_Vtbl = {QueryInterface,
AddRef,
Release,
Callback1,
Callback2,
Callback3,
Callback4,
Callback5};




int main(int argc, char **argv)
{
	IExampleEvts				*exampleObj;
	IConnectionPointContainer	*container;
	IConnectionPoint			*point;
	HRESULT						hr;
	DWORD						cookie, i;

	// Although we don't have to allocate our IFeedback, we do need to
	// initialize it, by storing its VTable in its lpVtbl member
	MyFeedback.lpVtbl = (IFeedbackVtbl *)&IFeedback_Vtbl;

	if (!CoInitialize(0))
	{
		// Get an IExampleEvts object
		if ((hr = CoCreateInstance(&CLSID_IExampleEvts, 0, CLSCTX_ALL, &IID_IExampleEvts, &exampleObj)))
			MessageBox(0, "Can't create IExampleEvts object", "CreateInstance error", MB_OK|MB_ICONEXCLAMATION);
		else
		{
			// Get IExampleEvts' IConnectionPointContainer sub-object. We do this by calling
			// IExampleEvts' QueryInterface, and pass it the standard GUID for an
			// IConnectionPointContainer VTable
			if ((hr = exampleObj->lpVtbl->QueryInterface(exampleObj, &IID_IConnectionPointContainer, &container)))
				MessageBox(0, "Can't get IConnectionPointContainer object", "QueryInterface error", MB_OK|MB_ICONEXCLAMATION);
			else
			{
				// Get IExampleEvts' IConnectionPoint sub-object for specifically giving
				// IExampleEvts our IFeedback. We do this by calling IConnectionPointContainer's
				// FindConnectionPoint, and pass it IFeedback VTable's GUID (defined for us in
				// IExampleEvts.h)
				hr = container->lpVtbl->FindConnectionPoint(container, &DIID_IFeedback, &point);

				// We don't need the IConnectionPointContainer, now that we got the one IConnectionPoint
				// we want (ie, the one we use to give IExampleEvts our IFeedback)
				container->lpVtbl->Release(container);

				if (hr)
					MessageBox(0, "Can't get IConnectionPoint object", "FindConnectionPoint error", MB_OK|MB_ICONEXCLAMATION);
				else
				{
					// Now call the IConnectionPoint's Advise function, giving it some object
					// whose QueryInterface it will call to get our IFeedback object. Let's
					// just give it our IFeedback object. So Advise will call our IFeedback's
					// QueryInterface to tell us to give it a pointer to our IFeedback. Dumb?
					// You bet. All of this convoluted stuff is a combination of poor pre-planning
					// by Microsoft programmers when they designed this stuff, as well as the
					// colossal blunder of designing COM to accomodate the limitations of early,
					// primitive editions of Visual Basic.
					//
					// Advise() gives us back a "cookie" value that we'll need to later pass to
					// Unadvise() (when we want to tell IExampleEvts to stop using our IFeedback)
					if ((hr = point->lpVtbl->Advise(point, (IUnknown *)&MyFeedback, &cookie)))
						MessageBox(0, "Can't set our IFeedback object", "Advise error", MB_OK|MB_ICONEXCLAMATION);
					else
					{
						// Call IExampleEvts' DoSomething() function 10 times. This is a contrived
						// example. Each time DoSomething() is called, it will simply call another
						// one of our IFeedback callback functions, in turn
						for (i=0; i < 10; i++)
							exampleObj->lpVtbl->DoSomething(exampleObj);

						// Tell IExampleEvts to stop using our IFeedback (and Release it). We pass
						// the cookie value we got from Advise() above
						if ((hr = point->lpVtbl->Unadvise(point, cookie)))
							MessageBox(0, "Can't unset our IFeedback object", "Unadvise error", MB_OK|MB_ICONEXCLAMATION);

						// Release the IConnectionPoint now that we're done with it (and
						// we've told IExampleEvts to stop calling our IFeedback functions)
						point->lpVtbl->Release(point);
					}
				}

			}

			// Release the IExampleEvts now that we're done with it
			exampleObj->lpVtbl->Release(exampleObj);
		}

		// When finally done with OLE, free it
		CoUninitialize();
	}
	else
		MessageBox(0, "Can't initialize COM", "CoInitialize error", MB_OK|MB_ICONEXCLAMATION);

	return(0);
}
